import cv2
import numpy as np
from omr_logic import OmrEngine
import easyocr

# Initialize EasyOCR Reader
print("Initializing EasyOCR...")
reader = easyocr.Reader(['fa', 'en'], gpu=False)

# Initialize OMR Engine
engine = OmrEngine(reader=reader)

# Path to the uploaded image
image_path = r"C:\Users\Hossein\.gemini\antigravity\brain\079ac12b-c92a-45ec-a3e5-e9f00221fc53\uploaded_image_1767005534415.jpg"

print(f"\nProcessing: {image_path}")
print("="*60)

# Load and resize image
image = cv2.imread(image_path)
if image is None:
    print("ERROR: Could not load image!")
    exit()

# Resize to standard width
target_width = 1000
h, w = image.shape[:2]
scale = target_width / w
target_height = int(h * scale)
image = cv2.resize(image, (target_width, target_height))

# Extract header for National ID
header_y2 = int(target_height * 0.25)
header_roi = image[0:header_y2, :]

# Save header for inspection
cv2.imwrite("static/debug_header.jpg", header_roi)
print(f"Saved header: static/debug_header.jpg")

# Try to extract National ID
print("\n" + "="*60)
print("NATIONAL ID EXTRACTION")
print("="*60)

national_id = engine.extract_national_id(header_roi)

print("\n" + "="*60)
print(f"RESULT: {national_id}")
print("="*60)

# Expected: 1367937401 (user confirmed)
expected_id = "1367937401"
if national_id == expected_id:
    print(f"✓ SUCCESS: Correctly extracted {national_id}")
else:
    print(f"✗ FAILED: Expected {expected_id}, got {national_id}")
    
    # Check if it's valid
    if national_id and engine.validate_iranian_national_id(national_id):
        print(f"  But {national_id} is a VALID Iranian National ID")
    else:
        print(f"  {national_id} is NOT a valid Iranian National ID")

print("\n" + "="*60)
print("Analyzing all OCR results in detail...")
print("="*60)

# Re-read the header with detailed output
results = reader.readtext(header_roi)
print(f"\nFound {len(results)} text regions:\n")

for idx, (bbox, text, prob) in enumerate(results):
    print(f"{idx+1}. Text: '{text}'")
    print(f"   Confidence: {prob:.3f}")
    print(f"   BBox: {bbox}")
    
    # Normalize and extract digits
    normalized = engine.normalize_digits(text)
    digits = ''.join(c for c in normalized if c.isdigit())
    
    if digits:
        print(f"   Digits: {digits}")
        
        # Check all 10-digit subsequences
        if len(digits) >= 10:
            for i in range(len(digits) - 9):
                candidate = digits[i:i+10]
                valid = engine.validate_iranian_national_id(candidate)
                print(f"     {candidate} -> {'✓ VALID' if valid else '✗ invalid'}")
    print()
