import cv2
import easyocr
import os

# Test image
image_path = r"C:/Users/Hossein/.gemini/antigravity/brain/6db23134-4202-4417-adc4-6d56a82f6983/uploaded_image_1766909449714.jpg"

print("Loading image...")
image = cv2.imread(image_path)
if image is None:
    print("ERROR: Can't load image!")
    exit(1)

h, w = image.shape[:2]
print(f"Original size: {w}x{h}")

# Resize to 1000px width
target_width = 1000
scale = target_width / w
target_height = int(h * scale)
image = cv2.resize(image, (target_width, target_height))
print(f"Resized to: {target_width}x{target_height}")

# Let's try different ROI strategies
print("\n=== Testing different ROI strategies ===\n")

# Strategy 1: Top 25% (full width)
roi1_h = int(target_height * 0.25)
roi1 = image[0:roi1_h, :]
cv2.imwrite("debug_roi1_top25_fullwidth.jpg", roi1)
print(f"1. Top 25% (full width): {roi1.shape[1]}x{roi1.shape[0]}")

# Strategy 2: Top-Right quadrant (current approach)
header_h = int(target_height * 0.25)
header_x_start = int(target_width * 0.4)
roi2 = image[0:header_h, header_x_start:]
cv2.imwrite("debug_roi2_topright.jpg", roi2)
print(f"2. Top-Right quadrant: {roi2.shape[1]}x{roi2.shape[0]}")

# Strategy 3: Just the blue box area (where National ID is)
# From your image, the blue box seems to be approximately:
# - Top: ~3-23% of height
# - Right side for ID: ~50-95% of width
roi3_y1 = int(target_height * 0.03)
roi3_y2 = int(target_height * 0.22)
roi3_x1 = int(target_width * 0.50)
roi3_x2 = int(target_width * 0.95)
roi3 = image[roi3_y1:roi3_y2, roi3_x1:roi3_x2]
cv2.imwrite("debug_roi3_bluebox_right.jpg", roi3)
print(f"3. Blue box (right side): {roi3.shape[1]}x{roi3.shape[0]}")

# Initialize EasyOCR
print("\n=== Initializing EasyOCR ===")
reader = easyocr.Reader(['fa', 'en'], gpu=False)

# Test each ROI
for idx, roi in enumerate([roi1, roi2, roi3], 1):
    print(f"\n--- Testing ROI {idx} ---")
    
    # Upscale
    scale_factor = 2
    width = int(roi.shape[1] * scale_factor)
    height = int(roi.shape[0] * scale_factor)
    enlarged = cv2.resize(roi, (width, height), interpolation=cv2.INTER_LINEAR)
    
    gray = cv2.cvtColor(enlarged, cv2.COLOR_BGR2GRAY)
    _, thresh = cv2.threshold(gray, 0, 255, cv2.THRESH_BINARY + cv2.THRESH_OTSU)
    
    # Save for inspection
    cv2.imwrite(f"debug_roi{idx}_thresh.jpg", thresh)
    
    # Run OCR with ONLY Persian digits
    results = reader.readtext(thresh, allowlist='۰۱۲۳۴۵۶۷۸۹')
    
    print(f"Found {len(results)} text regions:")
    for (bbox, text, prob) in results:
        if prob > 0.05:  # Only show reasonable confidence
            print(f"  '{text}' (confidence: {prob:.2f})")
            
            # Normalize to English digits
            normalized = text
            for persian, english in zip('۰۱۲۳۴۵۶۷۸۹', '0123456789'):
                normalized = normalized.replace(persian, english)
            
            if len(normalized) >= 10:
                print(f"  -> Normalized: '{normalized}' [CANDIDATE!]")

print("\n=== Check the saved debug images to see what's being read ===")
print("Saved images:")
print("  - debug_roi1_top25_fullwidth.jpg")
print("  - debug_roi2_topright.jpg")
print("  - debug_roi3_bluebox_right.jpg")
print("  - debug_roi1_thresh.jpg")
print("  - debug_roi2_thresh.jpg")
print("  - debug_roi3_thresh.jpg")
