"""
Test script for OMR Engine Flask API
Tests the /process-omr endpoint with a sample image
"""

import requests
import json
import os

# Configuration
API_URL = "http://localhost:5000/process-omr"
IMAGE_PATH = r"C:\Users\Hossein\.gemini\antigravity\brain\079ac12b-c92a-45ec-a3e5-e9f00221fc53\uploaded_image_1767005534415.jpg"

# Alternative: Use a local test image
# IMAGE_PATH = "temp_uploads/unnamed.jpg"

print("="*60)
print("🧪 Testing OMR Engine API")
print("="*60)

# Check if image exists
if not os.path.exists(IMAGE_PATH):
    print(f"❌ Error: Image not found at {IMAGE_PATH}")
    print("\nPlease update IMAGE_PATH in this script to point to a valid image.")
    exit(1)

print(f"📄 Image: {IMAGE_PATH}")
print(f"🌐 API: {API_URL}")
print("\n" + "="*60)

# Test 1: Health check
print("\n1️⃣ Testing health endpoint...")
try:
    response = requests.get("http://localhost:5000/health")
    print(f"   Status: {response.status_code}")
    print(f"   Response: {response.json()}")
except Exception as e:
    print(f"   ❌ Error: {e}")
    print("\n   Make sure Flask server is running:")
    print("   python app.py")
    exit(1)

# Test 2: Process OMR
print("\n2️⃣ Testing OMR processing...")
print("   Uploading image...")

try:
    with open(IMAGE_PATH, 'rb') as img_file:
        files = {'image': img_file}
        data = {'options_count': 5}  # 5 options per question
        
        response = requests.post(API_URL, files=files, data=data)
    
    print(f"   Status: {response.status_code}")
    
    if response.status_code == 200:
        result = response.json()
        
        print("\n✅ SUCCESS!")
        print("="*60)
        print(f"📋 National ID: {result.get('national_id')}")
        print(f"⏱️  Processing Time: {result.get('processing_time_ms')} ms")
        print(f"📊 Total Questions: {result.get('metadata', {}).get('total_questions')}")
        
        print("\n📝 Answers:")
        answers = result.get('answers', {})
        for q_num in sorted([int(k) for k in answers.keys()]):
            answer = answers[str(q_num)]
            print(f"   Q{q_num}: {answer if answer else 'No answer'}")
        
        print("\n🖼️  Debug Images:")
        debug_images = result.get('debug_images', {})
        for name, url in debug_images.items():
            print(f"   {name}: http://localhost:5000{url}")
        
        print("="*60)
        
        # Validation for the test image
        print("\n🔍 Validation:")
        expected_id = "1367937401"
        actual_id = result.get('national_id')
        
        if actual_id == expected_id:
            print(f"   ✅ National ID correct: {actual_id}")
        else:
            print(f"   ⚠️  National ID mismatch:")
            print(f"      Expected: {expected_id}")
            print(f"      Got:      {actual_id}")
    
    else:
        print(f"\n❌ ERROR: {response.status_code}")
        print(json.dumps(response.json(), indent=2, ensure_ascii=False))

except requests.exceptions.ConnectionError:
    print("\n❌ Connection Error!")
    print("   Make sure Flask server is running:")
    print("   python app.py")
except Exception as e:
    print(f"\n❌ Error: {e}")
    import traceback
    traceback.print_exc()

print("\n" + "="*60)
print("✅ Test completed")
print("="*60)
