"""
OMR Engine Configuration
Environment-based configuration for Flask app
"""

import os


class Config:
    """Base configuration"""
    
    # Flask
    SECRET_KEY = os.environ.get('SECRET_KEY') or 'dev-secret-key-change-in-production'
    DEBUG = False
    TESTING = False
    
    # File Upload
    UPLOAD_FOLDER = 'temp_uploads'
    STATIC_FOLDER = 'static'
    MAX_CONTENT_LENGTH = 10 * 1024 * 1024  # 10MB
    ALLOWED_EXTENSIONS = {'png', 'jpg', 'jpeg'}
    
    # OMR Engine
    DEFAULT_OPTIONS_COUNT = 5
    MIN_OPTIONS_COUNT = 2
    MAX_OPTIONS_COUNT = 10
    
    # EasyOCR
    OCR_LANGUAGES = ['fa', 'en']
    OCR_GPU = True  # Set to False if no GPU available
    
    # Image Processing
    TARGET_WIDTH = 1000  # Standardized image width
    HEADER_ROI_RATIO = 0.25  # Top 25% for National ID
    BODY_START_RATIO = 0.26  # Start of answer area


class DevelopmentConfig(Config):
    """Development configuration"""
    DEBUG = True
    OCR_GPU = False  # Usually CPU for development


class ProductionConfig(Config):
    """Production configuration"""
    DEBUG = False
    OCR_GPU = True  # Enable GPU in production


class TestingConfig(Config):
    """Testing configuration"""
    TESTING = True
    DEBUG = True
    OCR_GPU = False


# Configuration dictionary
config = {
    'development': DevelopmentConfig,
    'production': ProductionConfig,
    'testing': TestingConfig,
    'default': DevelopmentConfig
}


def get_config(env=None):
    """Get configuration based on environment"""
    if env is None:
        env = os.environ.get('FLASK_ENV', 'development')
    return config.get(env, config['default'])
