"""
OMR Engine - Flask API Server
Clean Architecture: Actions, Services, Repositories pattern
"""

from flask import Flask, request, jsonify, send_from_directory
from werkzeug.utils import secure_filename
import os
import easyocr
from omr_logic import OmrEngine
import traceback
from datetime import datetime

# Initialize Flask app
app = Flask(__name__)

# Configuration
UPLOAD_FOLDER = 'temp_uploads'
STATIC_FOLDER = 'static'
ALLOWED_EXTENSIONS = {'png', 'jpg', 'jpeg'}
MAX_FILE_SIZE = 10 * 1024 * 1024  # 10MB

app.config['UPLOAD_FOLDER'] = UPLOAD_FOLDER
app.config['MAX_CONTENT_LENGTH'] = MAX_FILE_SIZE

# Ensure folders exist
os.makedirs(UPLOAD_FOLDER, exist_ok=True)
os.makedirs(STATIC_FOLDER, exist_ok=True)

# Initialize EasyOCR Reader (singleton - load once on startup)
print("🔄 Initializing EasyOCR Reader...")
try:
    # Try GPU first, fallback to CPU if GPU not available
    reader = easyocr.Reader(['fa', 'en'], gpu=True)
    print("✅ EasyOCR initialized with GPU")
except Exception as e:
    print(f"⚠️  GPU not available, using CPU: {e}")
    reader = easyocr.Reader(['fa', 'en'], gpu=False)
    print("✅ EasyOCR initialized with CPU")

# Initialize OMR Engine
engine = OmrEngine(reader=reader)
print("✅ OMR Engine ready")


def allowed_file(filename):
    """Check if file extension is allowed"""
    return '.' in filename and filename.rsplit('.', 1)[1].lower() in ALLOWED_EXTENSIONS


@app.route('/health', methods=['GET'])
def health_check():
    """Health check endpoint"""
    return jsonify({
        'status': 'healthy',
        'service': 'OMR Engine',
        'timestamp': datetime.now().isoformat()
    }), 200


@app.route('/scan', methods=['POST'])
def scan():
    """
    OMR Scanning endpoint (Laravel compatible)
    
    Request (from Laravel OMRService):
        - file: UploadedFile (required) - JPG/PNG image file
        - options_count: int (optional, default: 5) - Number of answer options per question
    
    Response:
        {
            "status": "success",
            "national_id": "1367937401",
            "answers": {"1": 3, "2": 4, ...},
            "debug_info": "Processed successfully"
        }
    """
    start_time = datetime.now()
    
    try:
        # Validate file in request (Laravel sends as 'file')
        if 'file' not in request.files:
            return jsonify({
                'status': 'error',
                'message': 'No file provided'
            }), 400
        
        file = request.files['file']
        
        if file.filename == '':
            return jsonify({
                'status': 'error',
                'message': 'No file selected'
            }), 400
        
        if not allowed_file(file.filename):
            return jsonify({
                'status': 'error',
                'message': f'Invalid file type. Allowed: {", ".join(ALLOWED_EXTENSIONS)}'
            }), 400
        
        # Get options_count parameter (default: 5)
        options_count = int(request.form.get('options_count', 5))
        
        if options_count < 2 or options_count > 10:
            return jsonify({
                'status': 'error',
                'message': 'options_count must be between 2 and 10'
            }), 400
        
        # Save uploaded file with secure filename
        filename = secure_filename(file.filename)
        timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
        unique_filename = f"{timestamp}_{filename}"
        filepath = os.path.join(app.config['UPLOAD_FOLDER'], unique_filename)
        
        file.save(filepath)
        print(f"📄 Saved uploaded file: {filepath}")
        
        # Process the exam sheet
        print(f"🔄 Processing OMR with {options_count} options...")
        result = engine.process_exam(filepath, options_count=options_count)
        
        # Clean up uploaded file (optional - comment out if you want to keep files)
        # os.remove(filepath)
        
        # Return Laravel-compatible response
        if result['status'] == 'success':
            return jsonify(result), 200
        else:
            return jsonify(result), 500
    
    except ValueError as ve:
        return jsonify({
            'status': 'error',
            'message': f'Invalid parameter: {str(ve)}'
        }), 400
    
    except Exception as e:
        print(f"❌ Error processing OMR: {e}")
        traceback.print_exc()
        
        return jsonify({
            'status': 'error',
            'message': str(e),
            'traceback': traceback.format_exc()
        }), 500


@app.route('/process-omr', methods=['POST'])
def process_omr():
    """
    Legacy OMR processing endpoint (backward compatibility)
    Accepts both 'image' and 'file' field names
    Returns extended response format with metadata
    
    Request:
        - image OR file: UploadedFile (required) - JPG/PNG image file
        - options_count: int (optional, default: 5) - Number of answer options per question
    
    Response:
        {
            "status": "success",
            "national_id": "1367937401",
            "answers": {"1": 3, "2": 4, ...},
            "debug_images": {...},
            "processing_time_ms": 2341,
            "metadata": {...}
        }
    """
    start_time = datetime.now()
    
    try:
        # Support both 'image' (legacy) and 'file' (Laravel)
        file = request.files.get('image') or request.files.get('file')
        
        if not file:
            return jsonify({
                'status': 'error',
                'message': 'No image file provided'
            }), 400
        
        if file.filename == '':
            return jsonify({
                'status': 'error',
                'message': 'No file selected'
            }), 400
        
        if not allowed_file(file.filename):
            return jsonify({
                'status': 'error',
                'message': f'Invalid file type. Allowed: {", ".join(ALLOWED_EXTENSIONS)}'
            }), 400
        
        # Get options_count parameter (default: 5)
        options_count = int(request.form.get('options_count', 5))
        
        if options_count < 2 or options_count > 10:
            return jsonify({
                'status': 'error',
                'message': 'options_count must be between 2 and 10'
            }), 400
        
        # Save uploaded file with secure filename
        filename = secure_filename(file.filename)
        timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
        unique_filename = f"{timestamp}_{filename}"
        filepath = os.path.join(app.config['UPLOAD_FOLDER'], unique_filename)
        
        file.save(filepath)
        print(f"📄 Saved uploaded file: {filepath}")
        
        # Process the exam sheet
        print(f"🔄 Processing OMR with {options_count} options...")
        result = engine.process_exam(filepath, options_count=options_count)
        
        # Calculate processing time
        processing_time = (datetime.now() - start_time).total_seconds() * 1000
        
        # Add debug images URLs if they exist
        debug_images = {}
        if os.path.exists('static/debug_circles.jpg'):
            debug_images['circles'] = '/static/debug_circles.jpg'
        if os.path.exists('static/debug_gray.jpg'):
            debug_images['gray'] = '/static/debug_gray.jpg'
        
        # Clean up uploaded file (optional - comment out if you want to keep files)
        # os.remove(filepath)
        
        # Enhance response
        if result['status'] == 'success':
            return jsonify({
                'status': 'success',
                'national_id': result.get('national_id'),
                'answers': result.get('answers', {}),
                'debug_images': debug_images,
                'processing_time_ms': round(processing_time, 2),
                'metadata': {
                    'options_count': options_count,
                    'total_questions': len(result.get('answers', {})),
                    'timestamp': datetime.now().isoformat()
                }
            }), 200
        else:
            return jsonify({
                'status': 'error',
                'message': result.get('message', 'Unknown error'),
                'processing_time_ms': round(processing_time, 2)
            }), 500
    
    except ValueError as ve:
        return jsonify({
            'status': 'error',
            'message': f'Invalid parameter: {str(ve)}'
        }), 400
    
    except Exception as e:
        print(f"❌ Error processing OMR: {e}")
        traceback.print_exc()
        
        return jsonify({
            'status': 'error',
            'message': str(e),
            'traceback': traceback.format_exc()
        }), 500


@app.route('/static/<path:filename>')
def serve_static(filename):
    """Serve debug images from static folder"""
    return send_from_directory(STATIC_FOLDER, filename)


@app.route('/', methods=['GET'])
def index():
    """Welcome page"""
    return jsonify({
        'service': 'OMR Engine API',
        'version': '2.0',
        'endpoints': {
            'health': 'GET /health',
            'process_omr': 'POST /process-omr',
            'static_files': 'GET /static/<filename>'
        },
        'documentation': {
            'process_omr': {
                'method': 'POST',
                'content_type': 'multipart/form-data',
                'parameters': {
                    'image': 'file (required) - JPG/PNG',
                    'options_count': 'int (optional, default: 5) - Number of options per question'
                }
            }
        }
    }), 200


if __name__ == '__main__':
    print("\n" + "="*60)
    print("🚀 OMR Engine Flask Server")
    print("="*60)
    print("📍 Endpoints:")
    print("   - GET  /          - API information")
    print("   - GET  /health    - Health check")
    print("   - POST /scan      - Process OMR sheet (Laravel compatible)")
    print("   - POST /process-omr - Process OMR sheet (legacy)")
    print("   - GET  /static/<file> - Debug images")
    print("="*60)
    print("🔧 Configuration:")
    print(f"   - Upload folder: {UPLOAD_FOLDER}")
    print(f"   - Max file size: {MAX_FILE_SIZE // (1024*1024)}MB")
    print(f"   - Allowed types: {', '.join(ALLOWED_EXTENSIONS)}")
    print("="*60 + "\n")
    
    # Run Flask app
    # Development mode
    app.run(
        host='0.0.0.0',  # Accessible from network
        port=5000,       # Default Flask port
        debug=True       # Auto-reload on code changes
    )
