import cv2
import numpy as np

# Load test image
image_path = r"C:/Users/Hossein/.gemini/antigravity/brain/6db23134-4202-4417-adc4-6d56a82f6983/uploaded_image_1766909449714.jpg"

print("Loading image...")
image = cv2.imread(image_path)

# Resize
target_width = 1000
h, w = image.shape[:2]
scale = target_width / w
target_height = int(h * scale)
image = cv2.resize(image, (target_width, target_height))

# Extract answer area (below 26%)
body_start = int(target_height * 0.26)
body_roi = image[body_start:, :]

print(f"Body ROI size: {body_roi.shape}")

# Preprocessing
gray = cv2.cvtColor(body_roi, cv2.COLOR_BGR2GRAY)
blurred = cv2.GaussianBlur(gray, (5, 5), 0)
thresh = cv2.adaptiveThreshold(blurred, 255, cv2.ADAPTIVE_THRESH_GAUSSIAN_C, 
                               cv2.THRESH_BINARY_INV, 11, 2)

cv2.imwrite("test_body_thresh.jpg", thresh)
print("Saved: test_body_thresh.jpg")

# Find contours
contours, _ = cv2.findContours(thresh, cv2.RETR_EXTERNAL, cv2.CHAIN_APPROX_SIMPLE)
print(f"\nFound {len(contours)} total contours")

# Analyze ALL contours
print("\n=== ANALYZING ALL CONTOURS ===\n")

contour_data = []

for i, cnt in enumerate(contours):
    area = cv2.contourArea(cnt)
    
    if area < 50:  # Skip very tiny noise
        continue
    
    perimeter = cv2.arcLength(cnt, True)
    if perimeter == 0:
        continue
    
    circularity = 4 * np.pi * (area / (perimeter * perimeter))
    x, y, w, h = cv2.boundingRect(cnt)
    aspect_ratio = float(w) / h
    
    contour_data.append({
        'area': area,
        'circularity': circularity,
        'aspect_ratio': aspect_ratio,
        'w': w,
        'h': h,
        'x': x,
        'y': y
    })

# Sort by area
contour_data.sort(key=lambda c: c['area'], reverse=True)

# Show top 100
print("TOP 100 CONTOURS (by area):\n")
print(f"{'#':<4} {'Area':<8} {'Circ':<6} {'AR':<6} {'W':<5} {'H':<5} {'X':<5} {'Y':<5}")
print("-" * 55)

for i, c in enumerate(contour_data[:100]):
    print(f"{i+1:<4} {c['area']:<8.0f} {c['circularity']:<6.2f} {c['aspect_ratio']:<6.2f} "
          f"{c['w']:<5} {c['h']:<5} {c['x']:<5} {c['y']:<5}")

# Statistics
if contour_data:
    areas = [c['area'] for c in contour_data]
    circs = [c['circularity'] for c in contour_data]
    
    print(f"\n=== STATISTICS ===")
    print(f"Total contours (area > 50): {len(contour_data)}")
    print(f"Area range: {min(areas):.0f} - {max(areas):.0f}")
    print(f"Circularity range: {min(circs):.2f} - {max(circs):.2f}")
    print(f"Average area: {np.mean(areas):.0f}")
    print(f"Average circularity: {np.mean(circs):.2f}")
    
    # Count by area ranges
    print(f"\nArea distribution:")
    print(f"  50-100: {len([a for a in areas if 50 <= a < 100])}")
    print(f"  100-200: {len([a for a in areas if 100 <= a < 200])}")
    print(f"  200-400: {len([a for a in areas if 200 <= a < 400])}")
    print(f"  400-800: {len([a for a in areas if 400 <= a < 800])}")
    print(f"  800+: {len([a for a in areas if a >= 800])}")
    
    print(f"\nCircularity distribution:")
    print(f"  < 0.5: {len([c for c in circs if c < 0.5])}")
    print(f"  0.5-0.7: {len([c for c in circs if 0.5 <= c < 0.7])}")
    print(f"  0.7-0.8: {len([c for c in circs if 0.7 <= c < 0.8])}")
    print(f"  0.8-0.9: {len([c for c in circs if 0.8 <= c < 0.9])}")
    print(f"  0.9+: {len([c for c in circs if c >= 0.9])}")
