# 📋 راهنمای جامع اتصال سیستم تصحیح هوشمند (OMR)

## 🔍 تحلیل معماری فعلی

### جریان کامل سیستم:

```
┌─────────────────┐
│  React Admin    │  (Frontend)
│  Panel          │  http://localhost:3000
│  OMRScanner.jsx │
└────────┬────────┘
         │ POST /api/exam/scan
         │ POST /api/exam/keys
         ▼
┌─────────────────┐
│  Laravel API    │  (Backend)
│  Port: 8000     │
│  OMRController  │
│  OMRService     │
└────────┬────────┘
         │ POST /process-omr
         │ (باید تغییر کند به /process-omr)
         ▼
┌─────────────────┐
│  Python Flask   │  (OMR Engine)
│  Port: 5000     │
│  app.py         │
│  omr_logic.py   │
└─────────────────┘
         │
         ▼
┌─────────────────┐
│  MySQL DB       │
│  exam_results   │
│  answer_keys    │
└─────────────────┘
```

---

## ⚠️ مشکلات فعلی و راه‌حل‌ها

### 1️⃣ عدم تطابق API Endpoint

**مشکل:**
- Laravel OMRService در خط 43 به `/scan` درخواست می‌فرستد
- Flask در `app.py` endpoint `/process-omr` داره

**راه‌حل:**

**گزینه A:** تغییر Flask (توصیه می‌شود)
```python
# در app.py خط 62
@app.route('/scan', methods=['POST'])  # به جای /process-omr
def scan():
    # همان محتوای process_omr
```

**گزینه B:** تغییر Laravel
```php
// در OMRService.php خط 43
$targetUrl = "{$this->pythonApiUrl}/process-omr";
```

---

### 2️⃣ عدم تطابق نام فیلد آپلود

**مشکل:**
- Laravel در خط 55 فیلد را با نام `file` می‌فرستد
- Flask در `app.py` انتظار فیلد `image` دارد

**راه‌حل:**

**گزینه A:** تغییر Flask (توصیه می‌شود)
```python
# در app.py خط 66
if 'file' not in request.files:  # به جای 'image'
    return jsonify({...}), 400

file = request.files['file']  # به جای 'image'
```

**گزینه B:** تغییر Laravel
```php
// در OMRService.php خط 54
->attach(
    'image', // به جای 'file'
    file_get_contents($file->getRealPath()),
    $file->getClientOriginalName()
)
```

---

### 3️⃣ متغیر محیطی برای URL پایتون

**مشکل:**
در `.env` لاراول متغیر `PYTHON_OMR_URL` تعریف نشده

**راه‌حل:**
اضافه کردن به `.env`:
```env
PYTHON_OMR_URL=http://127.0.0.1:5000
OMR_TIMEOUT=600
```

---

## 🔧 تغییرات مورد نیاز

### 1. تغییرات در Flask (`app.py`)

```python
# تغییر endpoint و نام فیلد برای سازگاری با Laravel

@app.route('/scan', methods=['POST'])  # ✅ تغییر از /process-omr
def scan():
    """
    OMR Scanning endpoint compatible with Laravel
    
    Request (from Laravel):
        - file: UploadedFile (required)
        - options_count: int (optional, default: 5)
    
    Response:
        {
            "status": "success",
            "national_id": "1367937401",
            "answers": {...}
        }
    """
    start_time = datetime.now()
    
    try:
        # ✅ تغییر نام فیلد از 'image' به 'file'
        if 'file' not in request.files:
            return jsonify({
                'status': 'error',
                'message': 'No file provided'
            }), 400
        
        file = request.files['file']
        
        if file.filename == '':
            return jsonify({
                'status': 'error',
                'message': 'No file selected'
            }), 400
        
        # Get options_count
        options_count = int(request.form.get('options_count', 5))
        
        if options_count < 2 or options_count > 10:
            return jsonify({
                'status': 'error',
                'message': 'options_count must be between 2 and 10'
            }), 400
        
        # Save file
        filename = secure_filename(file.filename)
        timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
        unique_filename = f"{timestamp}_{filename}"
        filepath = os.path.join(app.config['UPLOAD_FOLDER'], unique_filename)
        
        file.save(filepath)
        print(f"📄 Saved: {filepath}")
        
        # Process
        print(f"🔄 Processing with {options_count} options...")
        result = engine.process_exam(filepath, options_count=options_count)
        
        # Clean up (optional)
        # os.remove(filepath)
        
        return jsonify(result), 200 if result['status'] == 'success' else 500
    
    except ValueError as ve:
        return jsonify({
            'status': 'error',
            'message': f'Invalid parameter: {str(ve)}'
        }), 400
    
    except Exception as e:
        print(f"❌ Error: {e}")
        traceback.print_exc()
        
        return jsonify({
            'status': 'error',
            'message': str(e)
        }), 500


# Keep /process-omr for backward compatibility
@app.route('/process-omr', methods=['POST'])
def process_omr():
    """Legacy endpoint - redirects to /scan"""
    return scan()
```

---

### 2. به‌روزرسانی `.env` لاراول

اضافه کنید:
```env
# OMR Python Microservice
PYTHON_OMR_URL=http://127.0.0.1:5000
OMR_TIMEOUT=600
```

---

### 3. به‌روزرسانی `.env.example` لاراول

```env
# OMR Python Microservice Configuration
PYTHON_OMR_URL=http://127.0.0.1:5000
OMR_TIMEOUT=600
```

---

## 📊 جدول Database

جدول `exam_results` از قبل وجود دارد و ساختار صحیحی دارد:

```sql
CREATE TABLE exam_results (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    racing_id BIGINT UNSIGNED NOT NULL,
    national_id VARCHAR(255) NOT NULL,
    user_id BIGINT UNSIGNED NULL,
    total_score DECIMAL(8,2) DEFAULT 0,
    correct_count INT DEFAULT 0,
    incorrect_count INT DEFAULT 0,
    unanswered_count INT DEFAULT 0,
    details JSON NULL,
    status VARCHAR(255) DEFAULT 'processed',
    image_path VARCHAR(255) NULL,
    created_at TIMESTAMP NULL,
    updated_at TIMESTAMP NULL,
    
    INDEX idx_national_id (national_id),
    INDEX idx_user_id (user_id),
    INDEX idx_racing_national (racing_id, national_id)
);
```

---

## 🚀 دستورالعمل راه‌اندازی کامل

### مرحله 1: راه‌اندازی Python OMR Engine

```bash
# 1. رفتن به پوشه OMR
cd i:/sin-fin/omr-engine

# 2. نصب dependencies (اگر نصب نشده)
pip install -r requirements.txt

# 3. اجرای سرور Flask
python app.py
```

**خروجی مورد انتظار:**
```
🔄 Initializing EasyOCR Reader...
✅ EasyOCR initialized with CPU
✅ OMR Engine ready

============================================================
🚀 OMR Engine Flask Server
============================================================
📍 Endpoints:
   - GET  /          - API information
   - GET  /health    - Health check
   - POST /scan      - Process OMR sheet (Laravel compatible)
   - POST /process-omr - Process OMR sheet (alternative)
   - GET  /static/<file> - Debug images
============================================================

 * Running on http://0.0.0.0:5000
```

---

### مرحله 2: راه‌اندازی Laravel API

```bash
# 1. رفتن به پوشه API
cd i:/sin-fin/api

# 2. به‌روزرسانی .env
# اضافه کردن:
PYTHON_OMR_URL=http://127.0.0.1:5000
OMR_TIMEOUT=600

# 3. اجرای سرور Laravel
php artisan serve --port=8000
```

---

### مرحله 3: راه‌اندازی React Admin Panel

```bash
# 1. رفتن به پوشه Admin
cd i:/sin-fin/sincap-app-admin

# 2. اجرا
npm run dev
```

---

### مرحله 4: تست اتصال

#### A. تست مستقیم Python API

```bash
curl http://localhost:5000/health
```

**پاسخ مورد انتظار:**
```json
{
  "status": "healthy",
  "service": "OMR Engine",
  "timestamp": "2025-12-29T19:00:00"
}
```

#### B. تست از طریق Laravel

```bash
# ایجاد یک فایل test_laravel_omr.php
php artisan tinker

# در tinker:
$service = app(\App\Services\OMRService::class);
$file = new \Illuminate\Http\UploadedFile(
    'path/to/test/image.jpg',
    'test.jpg',
    'image/jpeg',
    null,
    true
);
$result = $service->processImage($file, 5);
dd($result);
```

#### C. تست از طریق Panel Admin

1. باز کردن `http://localhost:3000` (یا پورت React)
2. رفتن به **تصحیح هوشمند** (`/smart-scan`)
3. انتخاب مسابقه
4. تنظیم کلید پاسخ‌ها
5. آپلود تصویر پاسخنامه

---

## 🔍 عیب‌یابی

### خطا: Connection refused

**علت:** Flask سرور در حال اجرا نیست

**راه‌حل:**
```bash
cd i:/sin-fin/omr-engine
python app.py
```

---

### خطا: 404 Not Found - /scan

**علت:** Endpoint در Flask با نام دیگری تعریف شده

**راه‌حل:** اطمینان از وجود:
```python
@app.route('/scan', methods=['POST'])
```

---

### خطا: No file provided

**علت:** عدم تطابق نام فیلد

**راه‌حل:** 
- Flask باید `file` را چک کند (نه `image`)
- یا Laravel باید `image` بفرستد (نه `file`)

---

### خطا: PYTHON_OMR_URL not set

**راه‌حل:**
```bash
# در .env لاراول
PYTHON_OMR_URL=http://127.0.0.1:5000
```

بعد از تغییر:
```bash
php artisan config:clear
php artisan cache:clear
```

---

## 📝 نمونه Response کامل

### 1. از Python به Laravel:

```json
{
  "status": "success",
  "national_id": "1367937401",
  "answers": {
    "1": 3,
    "2": 4,
    "3": 1,
    "4": 2,
    ...
  },
  "debug_info": "Processed successfully"
}
```

### 2. از Laravel به React (بعد از Grading):

```json
{
  "success": true,
  "data": {
    "status": "success",
    "national_id": "1367937401",
    "answers": {...},
    "grading_status": "success",
    "exam_result": {
      "id": 123,
      "racing_id": 5,
      "national_id": "1367937401",
      "user_id": 42,
      "total_score": 85.5,
      "correct_count": 17,
      "incorrect_count": 2,
      "unanswered_count": 1,
      "details": {
        "1": {
          "user_option": 3,
          "correct_option": 3,
          "status": "correct",
          "score": 5
        },
        ...
      },
      "status": "processed",
      "image_path": "exam_papers/20251229_140530_scan.jpg",
      "created_at": "2025-12-29T14:05:32",
      "updated_at": "2025-12-29T14:05:32"
    }
  }
}
```

---

## ✅ Checklist نهایی

- [ ] Python Flask سرور روی `http://localhost:5000` در حال اجراست
- [ ] Endpoint `/scan` در Flask تعریف شده
- [ ] نام فیلد در Flask `file` است (یا `image` در Laravel تغییر داده شده)
- [ ] متغیر `PYTHON_OMR_URL` در `.env` لاراول تعریف شده
- [ ] Laravel API روی `http://localhost:8000` در حال اجراست
- [ ] React Admin Panel روی `http://localhost:3000` در حال اجراست
- [ ] جدول `exam_results` در دیتابیس وجود دارد
- [ ] جدول `answer_keys` در دیتابیس وجود دارد
- [ ] تست sلامت Python API با موفقیت پاس شد
- [ ] تست کامل از Panel Admin انجام شد

---

## 🎯 نکات مهم

1. **ترتیب راه‌اندازی مهم است:**
   - اول: Python OMR Engine
   - دوم: Laravel API
   - سوم: React Admin Panel

2. **Timeout:**
   - پردازش هر برگه ~2-5 ثانیه طول می‌کشد
   - برای پردازش گروهی، timeout باید حداقل 60 ثانیه باشد

3. **حجم فایل:**
   - Laravel: حداکثر 5MB
   - Flask: حداکثر 10MB
   - در صورت نیاز در هر دو تنظیم کنید

4. **لاگ‌ها:**
   - Python: در Console
   - Laravel: `storage/logs/laravel.log`
   - React: Developer Console

---

**آخرین بروزرسانی:** 2025-12-29  
**نسخه:** 2.0  
**وضعیت:** آماده برای Production ✅
